/*
* Creation date : Tues Mar 03 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* export/import ECC key functions. 
*
* \version CE2_ECPKI_BUILD.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/
#include "CE2_ECPKI_BUILD.h"
#include "LLF_ECPKI_BUILD.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/


/**
****************************************************************
* Function Name: 
*  CE2_ECPKI_BuildPrivKey
*
* Inputs:
* @param DomainID [in] - The enumerator variable defines current EC domain.
* @param PrivKeyIn_ptr [in] - Pointer to private key data.
* @param PrivKeySizeInBytes [in] - Size of private key data in bytes.
* @param UserPrivKey_ptr [out] - Pointer to the private key structure. 
*                    This structure is used as input to the ECPKI 
*                    cryptographic primitives.
*                    Size of buffer must be not less than:
*                    Order size of the appropriative domain in bytes
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECPKI_BLD_PRIV_KEY_ILLEGAL_DOMAIN_ID_ERROR
*   - CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_IN_PTR_ERROR
*   - CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_PTR_ERROR
*
* \brief \b 
* Description:
*  Builds (imports) the user private key structure 
*  from an existing private key so that this structure 
*  can be used by other EC primitives.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECPKI_BuildPrivKey 
*     for building of private key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECPKI_BuildPrivKey(
			  CE2_ECPKI_DomainID_t      DomainID,	      /*in */   
				DxUint8_t			      *PrivKeyIn_ptr,     /*in*/
				DxUint32_t                 PrivKeySizeInBytes,/*in*/
				CE2_ECPKI_UserPrivKey_t  *UserPrivKey_ptr    /*out*/ )
{
  if ((DomainID < CE2_ECPKI_DomainID_EMPTY) || 
    (DomainID >= CE2_ECPKI_DomainID_OffMode))
    return CE2_ECPKI_BLD_PRIV_KEY_ILLEGAL_DOMAIN_ID_ERROR;

  if (PrivKeyIn_ptr == DX_NULL)
    return CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_IN_PTR_ERROR;

  if (UserPrivKey_ptr == DX_NULL)
    return CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_PTR_ERROR;

  UserPrivKey_ptr->DomainID = DomainID;

  return LLF_ECPKI_BuildPrivKey(DomainID,	PrivKeyIn_ptr, 
    PrivKeySizeInBytes, UserPrivKey_ptr);
} /* End of CE2_ECPKI_BuildPrivKey */

/**
****************************************************************
* Function Name: 
*  CE2_ECPKI_BuildPublKey
*
* Inputs:
* @param DomainID [in] - The enumerator variable defines current EC domain.
* @param PublKeyIn_ptr [in] - Pointer to public key data.
* @param PublKeySizeInBytes [in] - Size of public key data in bytes.
* @param UserPublKey_ptr [out] - Pointer to the public key structure. 
*                    This structure is used as input to the ECPKI 
*                    cryptographic primitives.
*                    Size of buffer must be not less than:
*					 2*ModSizeInBytes - for WM DRM,
*					 2*ModSizeInBytes+1 - for other domains.  
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECPKI_BLD_PUBL_KEY_ILLEGAL_DOMAIN_ID_ERROR
*   - CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_IN_PTR_ERROR
*   - CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_PTR_ERROR
*
* \brief \b 
* Description:
*  Builds (imports) the user public key structure 
*  from an existing public key so that this structure 
*  can be used by other EC primitives.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECPKI_BuildPublKey 
*     for building of public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECPKI_BuildPublKey(
			  CE2_ECPKI_DomainID_t      DomainID,	        /*in*/				
				DxUint8_t			      *PublKeyIn_ptr,       /*in*/
				DxUint32_t                 PublKeySizeInBytes,  /*in*/
				CE2_ECPKI_UserPublKey_t  *UserPublKey_ptr      /*out*/ )
{
  if ((DomainID < CE2_ECPKI_DomainID_EMPTY) || 
    (DomainID >= CE2_ECPKI_DomainID_OffMode))
    return CE2_ECPKI_BLD_PUBL_KEY_ILLEGAL_DOMAIN_ID_ERROR;

  if (PublKeyIn_ptr == DX_NULL)
    return CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_IN_PTR_ERROR;

  if (UserPublKey_ptr == DX_NULL)
    return CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_PTR_ERROR;

  UserPublKey_ptr->DomainID = DomainID;

  return LLF_ECPKI_BuildPublKey(DomainID,	PublKeyIn_ptr, 
    PublKeySizeInBytes, UserPublKey_ptr);
} /* End of CE2_ECPKI_BuildPublKey */

/**
****************************************************************
* Function Name: 
*  CE2_ECPKI_ExportPublKey
*
* Inputs:
* @param UserPublKey_ptr [in] - A pointer to the public key structure 
*                    (in little-endian form). 
* @param Compression [in] - An enumerator parameter: defines point compression.
* @param ExternPublKey_ptr [out] - A pointer to the exported public key structure 
*                      in big-endian and point compression as defined 
*                      by input parameter. 
*                      Size of buffer must be not less than:
*					   2*ModSizeInBytes - for WM DRM,
*					   2*ModSizeInBytes+1 - for other domains.   
* @param PublKeySizeInBytes [in/out] - A pointer to a variable for input size of 
*                       user passed buffer for public key and output 
*                       the size of converted public key in bytes.
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECPKI_EXPORT_PUBL_KEY_INVALID_USER_PUBL_KEY_PTR_ERROR 
*   - CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_COMPRESSION_MODE_ERROR
*   - CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_PTR_ERROR
*   - CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  Converts an existing public key in little-endian form to  
*  big-endian export form and initializes an appropriate structure 
*  of type CE2_ECPKI_PublKeyExtern_t.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECPKI_ExportPublKey
*     for exporting of public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECPKI_ExportPublKey(
   			CE2_ECPKI_UserPublKey_t *UserPublKey_ptr,     /*in*/
				CE2_ECPKI_PointCompression_t  Compression,         /*in*/
				DxUint8_t	*ExternPublKey_ptr,   /*out*/
				DxUint32_t *PublKeySizeInBytes   /*in/out*/ )
{
  if (UserPublKey_ptr == DX_NULL) 
    return CE2_ECPKI_EXPORT_PUBL_KEY_INVALID_USER_PUBL_KEY_PTR_ERROR;

  if (!(Compression == CE2_EC_PointCompressed || 
    Compression == CE2_EC_PointUncompressed ||
    Compression == CE2_EC_PointHybrid))
    return CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_COMPRESSION_MODE_ERROR;

  if (ExternPublKey_ptr == DX_NULL)
    return CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_PTR_ERROR;

  if (PublKeySizeInBytes == DX_NULL) 
    return CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_SIZE_PTR_ERROR;

  return LLF_ECPKI_ExportPublKey(UserPublKey_ptr, Compression, 
    ExternPublKey_ptr, PublKeySizeInBytes);
} /* End of CE2_ECPKI_ExportPublKey */

/**
****************************************************************
* Function Name: 
*  CE2_ECPKI_ExportPrivKey
*
* Inputs:
* @param UserPrivKey_ptr [in] - A pointer to the private key structure 
*                    (in little-endian form). 
* @param ExternPrivKey_ptr [out] - A pointer to the exported private key 
*                      in big-endian. 
*						Size of buffer must be not less than:
*						Order size of the appropriative domain in bytes
* @param ExternPrivKeySize_ptr [in\out]- A pointer to a variable for 
*                       input size of user passed buffer for private key 
*                       and output the size of converted private key in bytes.
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_USER_PRIV_KEY_PTR_ERROR
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_PTR_ERROR
*   - CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  Converts an existing private key in little-endian form to  
*  big-endian export form.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECPKI_ExportPrivKey
*     for exporting of private key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECPKI_ExportPrivKey(
  CE2_ECPKI_UserPrivKey_t       *UserPrivKey_ptr,     /*in*/
  DxUint8_t			                *ExternPrivKey_ptr,   /*out*/
  DxUint32_t                    *ExternPrivKeySize_ptr   /*in/out*/ )
{
  if (UserPrivKey_ptr == DX_NULL)
    return CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_USER_PRIV_KEY_PTR_ERROR;

  if (ExternPrivKey_ptr == DX_NULL)
    return CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_PTR_ERROR;

  if (ExternPrivKeySize_ptr == DX_NULL)
    return CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_SIZE_PTR_ERROR;

  return LLF_ECPKI_ExportPrivKey(UserPrivKey_ptr, 
    ExternPrivKey_ptr, ExternPrivKeySize_ptr);
}
